package com.wakaleo.jpt.taxcalculator.impl;

import static org.junit.Assert.*;
import static org.hamcrest.Matchers.*;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

import org.joda.time.DateTime;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;

import com.wakaleo.jpt.taxcalculator.impl.TaxCalculatorImpl;
import com.wakaleo.jpt.taxcalculator.InvalidYearException;

public class TaxCalculatorImplTest {

    TaxCalculatorImpl taxCalculator = null;
    
    @Before
    public void prepareTaxCalculator() {
        taxCalculator = new TaxCalculatorImpl();
    }

    @After
    public void cleanupTaxCalculator() {
        taxCalculator = null;
    }

    @Test
    public void shouldUseLowestTaxRateForIncomeBelow38000() throws InvalidYearException {
        double income = 30000;
        double expectedTax = income * 0.195;
        double calculatedTax = taxCalculator.calculateIncomeTax(30000, 2008);
        assertThat(calculatedTax, is(expectedTax));
    }
        
    @Test(expected=InvalidYearException.class)
    public void futureYearsShouldBeInvalid() throws InvalidYearException {
        DateTime today = new DateTime();
        int nextYear = today.getYear() + 1;
        double income = 30000;
        taxCalculator.calculateIncomeTax(income, nextYear);
    }
    
    @Test
    public void exceptionShouldIncludeAClearMessage() throws InvalidYearException {
        double income = 30000;
        try {
            taxCalculator.calculateIncomeTax(income, 2100);
            fail("calculateIncomeTax() should have thrown an exception.");
        } catch (InvalidYearException expected) {
            assertEquals(expected.getMessage(),
                         "No tax calculations available yet for the year 2100");
        }
    }
    
    @Test
    public void hamcrestAssertEqualToRed(){
        String color = "red";
        assertThat(color, equalTo("red"));
    }
    
    @Test
    public void hamcrestAssertIsRed(){
        String color = "red";
        assertThat(color, is("red"));
    }
    
    @Test
    public void hamcrestAssertIsRedOrBLue(){
        String color = "red";
        assertThat(color, isOneOf("red","blue"));
        int i = 20;
        
    }

    @Test
    public void hamcrestAssertNotBlue(){
        String color = "red";
        assertThat(color, not("blue"));
    }

    @Test
    public void hamcrestAssertNotNull(){
        String color = "red";
        assertThat(color, is(notNullValue()));
    }
    

    @Test
    public void hamcrestAssertNull(){
        String color = null;
        assertThat(color, is(nullValue()));
    }

    @Test
    public void hamcrestAssertHasItem(){
        List<String> colors = new ArrayList<String>();
        colors.add("red");
        colors.add("green");
        colors.add("blue");
        assertThat(colors, hasItem("blue"));        
    }

    @Test
    public void hamcrestAssertHasItems(){
        List<String> colors = new ArrayList<String>();
        colors.add("red");
        colors.add("green");
        colors.add("blue");
        assertThat(colors, hasItems("red","blue"));        
        assertThat("red", isIn(colors));        
    }
    
    @Test
    public void hamcrestAssertHasItemInArray(){
        String[] colors = new String[] {"red","green","blue"};
        assertThat(colors, hasItemInArray("blue"));        
    }   
   
    @Test
    public void hamcrestAssertIsIn(){
        String[] colors = new String[] {"red","green","blue"};
        assertThat("red", isIn(colors));        
    }

    
    @Test
    public void hamcrestAssertIsRedOrBlue(){
        String color = "red";        
        assertThat(color, anyOf(is("red"),(is("blue"))));        
        assertThat(color,is(String.class));
    }

    @Test
    public void hamcrestIsClass(){
        List myList = new ArrayList();     
        assertThat(myList,is(Collection.class));
    }
    
    @Test
    public void hamcrestAssertIsBetween10and20(){
        int value = 15;       
        assertThat(value, allOf(greaterThanOrEqualTo(10),
                                lessThanOrEqualTo(20)));
        
        assertTrue("Value should be between 10 and 20",
                   value >= 10 && value <= 20);
    }
    
    
    @Test
    public void hamcrestAssertNotIsIn(){
        String[] colors = new String[] {"red","green","blue"};
        String color = "yellow";
        assertThat(color, not(isIn(colors)));        
    }       

    public void hamcrestHasItemLessThan(){
        List<Integer> ages = new ArrayList<Integer>();
        ages.add(20);
        ages.add(30);
        ages.add(40);
        assertThat(ages, not(hasItem(lessThan(18))));
    }
}